<?php
// generar_pdf_operaciones.php
require_once 'tcpdf/tcpdf.php';

// Configuración de zona horaria
date_default_timezone_set('America/Lima');

// Obtener parámetros
$tipo = isset($_GET['tipo']) ? $_GET['tipo'] : 'general';
$fecha_inicio = isset($_GET['start']) ? $_GET['start'] : null;
$fecha_fin = isset($_GET['end']) ? $_GET['end'] : null;
$producto_id = isset($_GET['producto']) ? $_GET['producto'] : null;

// Limpiar parámetros vacíos
if (empty($fecha_inicio)) $fecha_inicio = null;
if (empty($fecha_fin)) $fecha_fin = null;
if (empty($producto_id)) $producto_id = null;

// Crear instancia de TCPDF con orientación horizontal
$pdf = new TCPDF('L', PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);

// Configurar información del documento
$pdf->SetCreator(PDF_CREATOR);
$pdf->SetAuthor('Sistema de Inventario');
$pdf->SetTitle('Reporte de Movimientos - Compras y Ventas');
$pdf->SetSubject('Reporte de Operaciones');
$pdf->SetKeywords('TCPDF, PDF, reporte, compras, ventas, inventario');

// Configurar márgenes
$pdf->SetMargins(10, 15, 10);
$pdf->SetHeaderMargin(5);
$pdf->SetFooterMargin(10);

// Configurar salto de página automático
$pdf->SetAutoPageBreak(TRUE, 15);

// Configurar fuente
$pdf->SetFont('helvetica', '', 8);

// Agregar página
$pdf->AddPage();

// Título del reporte
$titulo = 'REPORTE DE MOVIMIENTOS - COMPRAS Y VENTAS';
if ($tipo === 'filtrado' && ($fecha_inicio || $fecha_fin || $producto_id)) {
    $titulo .= ' (FILTRADO)';
    $filtros_aplicados = array();
    
    if ($fecha_inicio && $fecha_fin) {
        $filtros_aplicados[] = "Período: " . date('d/m/Y', strtotime($fecha_inicio)) . ' al ' . date('d/m/Y', strtotime($fecha_fin));
    } elseif ($fecha_inicio) {
        $filtros_aplicados[] = "Desde: " . date('d/m/Y', strtotime($fecha_inicio));
    } elseif ($fecha_fin) {
        $filtros_aplicados[] = "Hasta: " . date('d/m/Y', strtotime($fecha_fin));
    }
    
    if ($producto_id) {
        $producto = ArticuloData::getById($producto_id);
        if ($producto) {
            $filtros_aplicados[] = "Producto: " . $producto->nombre;
        }
    }
    
    if (!empty($filtros_aplicados)) {
        $titulo .= "\n" . implode(" | ", $filtros_aplicados);
    }
}

$pdf->SetFont('helvetica', 'B', 14);
$pdf->Cell(0, 10, $titulo, 0, 1, 'C');
$pdf->Ln(5);

// Fecha de generación
$pdf->SetFont('helvetica', '', 9);
$pdf->Cell(0, 5, 'Generado el: ' . date('d/m/Y H:i:s'), 0, 1, 'R');
$pdf->Ln(5);

// Obtener datos según el tipo
if ($tipo === 'filtrado') {
    $operaciones = CompraData::vercontenido_reporte_nuevo($fecha_inicio, $fecha_fin, $producto_id);
} else {
    $operaciones = CompraData::vercontenido_reporte_nuevo();
}

if (empty($operaciones)) {
    $pdf->SetFont('helvetica', 'I', 12);
    $pdf->Cell(0, 10, 'No se encontraron datos para mostrar', 0, 1, 'C');
} else {
    // Función para dibujar encabezados
    function dibujarEncabezados($pdf, $col_widths) {
        $pdf->SetFont('helvetica', 'B', 7);
        $pdf->SetFillColor(52, 58, 64); // Color gris oscuro
        $pdf->SetTextColor(255, 255, 255); // Texto blanco
        
        $pdf->Cell($col_widths['fecha'], 8, 'Fecha', 1, 0, 'C', true);
        $pdf->Cell($col_widths['tipo'], 8, 'Tipo', 1, 0, 'C', true);
        $pdf->Cell($col_widths['articulo'], 8, 'Artículos', 1, 0, 'C', true);
        $pdf->Cell($col_widths['cantidad'], 8, 'Cantidad', 1, 0, 'C', true);
        $pdf->Cell($col_widths['saldo'], 8, 'Saldo', 1, 0, 'C', true);
        $pdf->Cell($col_widths['stock'], 8, 'Stock', 1, 0, 'C', true);
        $pdf->Cell($col_widths['destino'], 8, 'Destino', 1, 0, 'C', true);
        $pdf->Cell($col_widths['precio_unit'], 8, 'Precio Unit.', 1, 0, 'C', true);
        $pdf->Cell($col_widths['precio_total'], 8, 'Precio Total', 1, 0, 'C', true);
        $pdf->Cell($col_widths['subtotal'], 8, 'Subtotal', 1, 0, 'C', true);
        $pdf->Cell($col_widths['descuento'], 8, 'Descuento', 1, 0, 'C', true);
        $pdf->Cell($col_widths['igv'], 8, 'IGV', 1, 0, 'C', true);
        $pdf->Cell($col_widths['total'], 8, 'Total', 1, 1, 'C', true);
        
        $pdf->SetTextColor(0, 0, 0);
    }
    
    // Definir anchos de columnas
    $col_widths = array(
        'fecha' => 20,
        'tipo' => 18,
        'articulo' => 35,
        'cantidad' => 20,
        'saldo' => 20,
        'stock' => 20,
        'destino' => 25,
        'precio_unit' => 22,
        'precio_total' => 24,
        'subtotal' => 20,
        'descuento' => 20,
        'igv' => 18,
        'total' => 20
    );
    
    // Encabezados iniciales
    dibujarEncabezados($pdf, $col_widths);
    
    $total_compras = 0;
    $total_ventas = 0;
    $contador_operaciones = 0;
    $operaciones_mostradas = 0;
    
    foreach ($operaciones as $operacion) {
        $id_operacion = $operacion->id;
        $tipo_operacion = $operacion->tipo_operacion;
        $detalles = ProcesoData::vercontenidos_nuevo($id_operacion, $tipo_operacion, $producto_id);
        $num_articulos = count($detalles);
        
        // Si hay filtro por producto y no hay detalles que coincidan, saltar esta operación
        if ($producto_id && $num_articulos == 0) {
            continue;
        }
        
        // Incrementar contador de operaciones mostradas
        $operaciones_mostradas++;
        
        // Contadores para resumen
        $contador_operaciones++;
        if ($tipo_operacion === 'COMPRA') {
            $total_compras += $operacion->total;
        } else {
            $total_ventas += $operacion->total;
        }
        
        // Color de fondo alternado por operación
        $fill_color = ($operaciones_mostradas % 2 == 0) ? array(248, 249, 250) : array(255, 255, 255);
        
        if ($num_articulos > 0) {
            foreach ($detalles as $index => $detalle) {
                // Verificar si necesitamos una nueva página
                if ($pdf->GetY() > 180) {
                    $pdf->AddPage();
                    dibujarEncabezados($pdf, $col_widths);
                }
                
                $pdf->SetFont('helvetica', '', 6);
                $cantidad_final = $detalle->cantidad / $detalle->cantidad_unidad;
                
                // Calcular saldo y stock
                $saldo_anterior = ProcesoData::obtenerSaldoAnteriorPorAlmacen_nuevo(
                    $detalle->articulo, 
                    $detalle->almacen, 
                    $operacion->fecha,
                    $tipo_operacion,
                    $id_operacion
                );
                
                $stock_actual = ProcesoData::calcularStockDespuesOperacion(
                    $saldo_anterior, 
                    $cantidad_final, 
                    $tipo_operacion
                );
                
                $precio_total_articulo = $detalle->precio * $cantidad_final;
                
                // Configurar color de fondo
                $pdf->SetFillColor($fill_color[0], $fill_color[1], $fill_color[2]);
                
                // Calcular altura de la fila para combinar celdas
                $row_height = 6;
                
                // Fecha (combinada para todos los artículos de la operación)
                if ($index == 0) {
                    // Guardar posición actual
                    $current_y = $pdf->GetY();
                    $current_x = $pdf->GetX();
                    
                    // Calcular altura total necesaria para la operación
                    $total_height = $num_articulos * $row_height;
                    
                    // Dibujar celda combinada para fecha
                    $pdf->MultiCell($col_widths['fecha'], $total_height, date('d/m/Y', strtotime($operacion->fecha)), 1, 'C', true, 0, '', '', true, 0, false, true, $total_height, 'M');
                    
                    // Dibujar celda combinada para tipo
                    $tipo_color = ($tipo_operacion === 'COMPRA') ? array(40, 167, 69) : array(220, 53, 69);
                    $pdf->SetTextColor($tipo_color[0], $tipo_color[1], $tipo_color[2]);
                    $pdf->SetFont('helvetica', 'B', 6);
                    $pdf->MultiCell($col_widths['tipo'], $total_height, $tipo_operacion, 1, 'C', true, 0, '', '', true, 0, false, true, $total_height, 'M');
                    $pdf->SetTextColor(0, 0, 0);
                    $pdf->SetFont('helvetica', '', 6);
                    
                    // Volver a la posición original para continuar con las otras columnas
                    $pdf->SetXY($current_x + $col_widths['fecha'] + $col_widths['tipo'], $current_y);
                } else {
                    // Para las filas siguientes, saltar las columnas de fecha y tipo
                    $pdf->SetX($pdf->GetX() + $col_widths['fecha'] + $col_widths['tipo']);
                }
                
                // Datos del artículo
                $nombre_articulo = strlen($detalle->articulos) > 25 ? substr($detalle->articulos, 0, 22) . '...' : $detalle->articulos;
                $pdf->Cell($col_widths['articulo'], $row_height, $nombre_articulo, 1, 0, 'L', true);
                $pdf->Cell($col_widths['cantidad'], $row_height, number_format($cantidad_final) . ' ' . substr($detalle->unidades, 0, 8), 1, 0, 'C', true);
                $pdf->Cell($col_widths['saldo'], $row_height, number_format($saldo_anterior) . ' ' . substr($detalle->unidades, 0, 8), 1, 0, 'C', true);
                
                // Stock (en rojo si es negativo)
                if ($stock_actual < 0) {
                    $pdf->SetTextColor(220, 53, 69);
                    $pdf->SetFont('helvetica', 'B', 6);
                }
                $pdf->Cell($col_widths['stock'], $row_height, number_format($stock_actual) . ' ' . substr($detalle->unidades, 0, 8), 1, 0, 'C', true);
                if ($stock_actual < 0) {
                    $pdf->SetTextColor(0, 0, 0);
                    $pdf->SetFont('helvetica', '', 6);
                }
                
                $nombre_almacen = strlen($detalle->almacenes) > 18 ? substr($detalle->almacenes, 0, 15) . '...' : $detalle->almacenes;
                $pdf->Cell($col_widths['destino'], $row_height, $nombre_almacen, 1, 0, 'C', true);
                $pdf->Cell($col_widths['precio_unit'], $row_height, $operacion->simbolo . '. ' . number_format($detalle->precio, 2), 1, 0, 'R', true);
                $pdf->Cell($col_widths['precio_total'], $row_height, $operacion->simbolo . '. ' . number_format($precio_total_articulo, 2), 1, 0, 'R', true);
                
                // Totales (combinados para toda la operación)
                if ($index == 0) {
                    // Guardar posición actual
                    $current_y = $pdf->GetY();
                    $current_x = $pdf->GetX();
                    
                    // Dibujar celdas combinadas para los totales
                    $pdf->MultiCell($col_widths['subtotal'], $total_height, $operacion->simbolo . '. ' . number_format($operacion->subtotal, 2), 1, 'R', true, 0, '', '', true, 0, false, true, $total_height, 'M');
                    $pdf->MultiCell($col_widths['descuento'], $total_height, $operacion->simbolo . '. ' . number_format($operacion->descuento, 2), 1, 'R', true, 0, '', '', true, 0, false, true, $total_height, 'M');
                    $pdf->MultiCell($col_widths['igv'], $total_height, $operacion->simbolo . '. ' . number_format($operacion->igv, 2), 1, 'R', true, 0, '', '', true, 0, false, true, $total_height, 'M');
                    $pdf->MultiCell($col_widths['total'], $total_height, $operacion->simbolo . '. ' . number_format($operacion->total, 2), 1, 'R', true, 1, '', '', true, 0, false, true, $total_height, 'M');
                    
                    // Si no es el último artículo, volver a la siguiente línea
                    if ($index < $num_articulos - 1) {
                        $pdf->SetXY(10, $current_y + $row_height);
                    }
                } else {
                    // Para las filas siguientes, saltar las columnas de totales
                    $pdf->SetX($pdf->GetX() + $col_widths['subtotal'] + $col_widths['descuento'] + $col_widths['igv'] + $col_widths['total']);
                    $pdf->Ln();
                }
            }
            
            // Línea separadora entre operaciones (más sutil)
            $pdf->SetDrawColor(200, 200, 200);
            $pdf->Line(10, $pdf->GetY(), 287, $pdf->GetY());
            $pdf->SetDrawColor(0, 0, 0);
            $pdf->Ln(0);
            
        } else {
            // Operación sin artículos (solo cuando no hay filtro de producto)
            if (!$producto_id) {
                $pdf->SetFont('helvetica', '', 6);
                $pdf->SetFillColor($fill_color[0], $fill_color[1], $fill_color[2]);
                
                $pdf->Cell($col_widths['fecha'], 6, date('d/m/Y', strtotime($operacion->fecha)), 1, 0, 'C', true);
                
                $tipo_color = ($tipo_operacion === 'COMPRA') ? array(40, 167, 69) : array(220, 53, 69);
                $pdf->SetTextColor($tipo_color[0], $tipo_color[1], $tipo_color[2]);
                $pdf->SetFont('helvetica', 'B', 6);
                $pdf->Cell($col_widths['tipo'], 6, $tipo_operacion, 1, 0, 'C', true);
                $pdf->SetTextColor(0, 0, 0);
                $pdf->SetFont('helvetica', '', 6);
                
                $pdf->Cell($col_widths['articulo'], 6, 'Sin artículos', 1, 0, 'C', true);
                $pdf->Cell($col_widths['cantidad'], 6, '-', 1, 0, 'C', true);
                $pdf->Cell($col_widths['saldo'], 6, '-', 1, 0, 'C', true);
                $pdf->Cell($col_widths['stock'], 6, '-', 1, 0, 'C', true);
                $pdf->Cell($col_widths['destino'], 6, '-', 1, 0, 'C', true);
                $pdf->Cell($col_widths['precio_unit'], 6, '-', 1, 0, 'C', true);
                $pdf->Cell($col_widths['precio_total'], 6, '-', 1, 0, 'C', true);
                $pdf->Cell($col_widths['subtotal'], 6, $operacion->simbolo . '. ' . number_format($operacion->subtotal, 2), 1, 0, 'R', true);
                $pdf->Cell($col_widths['descuento'], 6, $operacion->simbolo . '. ' . number_format($operacion->descuento, 2), 1, 0, 'R', true);
                $pdf->Cell($col_widths['igv'], 6, $operacion->simbolo . '. ' . number_format($operacion->igv, 2), 1, 0, 'R', true);
                $pdf->Cell($col_widths['total'], 6, $operacion->simbolo . '. ' . number_format($operacion->total, 2), 1, 1, 'R', true);
            }
        }
    }
    
    // Resumen final con manejo de múltiples monedas
    $pdf->Ln(5);
    $pdf->SetFont('helvetica', 'B', 9);
    $pdf->SetFillColor(52, 58, 64);
    $pdf->SetTextColor(255, 255, 255);
    $pdf->Cell(100, 8, 'RESUMEN DE OPERACIONES', 1, 1, 'C', true);
    
    $pdf->SetTextColor(0, 0, 0);
    $pdf->SetFont('helvetica', '', 8);
    $pdf->SetFillColor(248, 249, 250);
    
    $pdf->Cell(70, 6, 'Total de Operaciones:', 1, 0, 'L', true);
    $pdf->Cell(30, 6, number_format($operaciones_mostradas), 1, 1, 'R', true);
    
    // Agrupar totales por moneda para compras
    $compras_por_moneda = array();
    $ventas_por_moneda = array();
    
    // Recalcular agrupando por moneda
    foreach ($operaciones as $operacion) {
        $id_operacion = $operacion->id;
        $tipo_operacion = $operacion->tipo_operacion;
        $detalles = ProcesoData::vercontenidos_nuevo($id_operacion, $tipo_operacion, $producto_id);
        $num_articulos = count($detalles);
        
        // Si hay filtro por producto y no hay detalles que coincidan, saltar esta operación
        if ($producto_id && $num_articulos == 0) {
            continue;
        }
        
        $moneda = $operacion->simbolo;
        
        if ($tipo_operacion === 'COMPRA') {
            if (!isset($compras_por_moneda[$moneda])) {
                $compras_por_moneda[$moneda] = 0;
            }
            $compras_por_moneda[$moneda] += $operacion->total;
        } else {
            if (!isset($ventas_por_moneda[$moneda])) {
                $ventas_por_moneda[$moneda] = 0;
            }
            $ventas_por_moneda[$moneda] += $operacion->total;
        }
    }
    
    // Mostrar totales de compras por moneda
    if (!empty($compras_por_moneda)) {
        $pdf->SetTextColor(40, 167, 69);
        $pdf->SetFont('helvetica', 'B', 8);
        $pdf->Cell(70, 6, 'TOTALES EN COMPRAS:', 1, 0, 'L', true);
        $pdf->Cell(30, 6, '', 1, 1, 'R', true);
        
        $pdf->SetFont('helvetica', '', 8);
        foreach ($compras_por_moneda as $moneda => $total) {
            $pdf->Cell(70, 6, '  ' . $moneda . ':', 1, 0, 'L', true);
            $pdf->Cell(30, 6, $moneda . '. ' . number_format($total, 2), 1, 1, 'R', true);
        }
    }
    
    // Mostrar totales de ventas por moneda
    if (!empty($ventas_por_moneda)) {
        $pdf->SetTextColor(220, 53, 69);
        $pdf->SetFont('helvetica', 'B', 8);
        $pdf->Cell(70, 6, 'TOTALES EN VENTAS:', 1, 0, 'L', true);
        $pdf->Cell(30, 6, '', 1, 1, 'R', true);
        
        $pdf->SetFont('helvetica', '', 8);
        foreach ($ventas_por_moneda as $moneda => $total) {
            $pdf->Cell(70, 6, '  ' . $moneda . ':', 1, 0, 'L', true);
            $pdf->Cell(30, 6, $moneda . '. ' . number_format($total, 2), 1, 1, 'R', true);
        }
    }
    
    $pdf->SetTextColor(0, 0, 0);
    
    // Mostrar información de filtros aplicados en el resumen
    if ($producto_id || $fecha_inicio || $fecha_fin) {
        $pdf->Ln(3);
        $pdf->SetFont('helvetica', 'B', 8);
        $pdf->SetFillColor(173, 216, 230);
        $pdf->Cell(100, 6, 'FILTROS APLICADOS', 1, 1, 'C', true);
        
        $pdf->SetFont('helvetica', '', 7);
        $pdf->SetFillColor(240, 248, 255);
        
        if ($fecha_inicio || $fecha_fin) {
            $fecha_filtro = '';
            if ($fecha_inicio && $fecha_fin) {
                $fecha_filtro = 'Período: ' . date('d/m/Y', strtotime($fecha_inicio)) . ' - ' . date('d/m/Y', strtotime($fecha_fin));
            } elseif ($fecha_inicio) {
                $fecha_filtro = 'Desde: ' . date('d/m/Y', strtotime($fecha_inicio));
            } elseif ($fecha_fin) {
                $fecha_filtro = 'Hasta: ' . date('d/m/Y', strtotime($fecha_fin));
            }
            $pdf->Cell(100, 5, $fecha_filtro, 1, 1, 'L', true);
        }
        
        if ($producto_id) {
            $producto = ArticuloData::getById($producto_id);
            if ($producto) {
                $nombre_producto = strlen($producto->nombre) > 80 ? substr($producto->nombre, 0, 77) . '...' : $producto->nombre;
                $pdf->Cell(100, 5, 'Producto: ' . $nombre_producto, 1, 1, 'L', true);
            }
        }
    }
}

// Configurar nombre del archivo
$filename = 'reporte_operaciones_' . date('Y-m-d_H-i-s');
if ($tipo === 'filtrado') {
    $filename .= '_filtrado';
}
if ($producto_id) {
    $filename .= '_producto';
}
$filename .= '.pdf';

// Generar PDF
$pdf->Output($filename, 'I');
?>